<?php

namespace App\Services\Website;

use App\Models\ContactUs;
use App\Models\Setting;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class ContactUsService
{
    public function store($data)
    {
        DB::beginTransaction();

        try {
            // Create the ContactUs
            $contactUs = ContactUs::create($data);

            // Send email notification
            $this->sendEmailNotification($contactUs);

            DB::commit();

            return true;
        } catch (\Exception $e) {

            DB::rollBack();

            throw $e;
        }
    }

    /**
     * Send email notification about new contact message
     */
    protected function sendEmailNotification(ContactUs $contactUs)
    {
        try {
            // Get admin email from settings or use default
            $adminEmail = Setting::where('key', 'site_email')
                ->where('lang', 'all')
                ->first()?->value ?? config('mail.from.address');

            if (!$adminEmail) {
                Log::warning('No admin email configured for contact form notifications');
                return;
            }

            // Prepare email data
            $name = $contactUs->name ?? 'غير محدد';
            $phone = $contactUs->phone ?? 'غير محدد';
            $email = $contactUs->email ?? 'غير محدد';
            $company = $contactUs->company ?? 'غير محدد';
            $message = $contactUs->message ?? 'غير محدد';

            // Get service name if service_id exists
            $service = 'غير محدد';
            if ($contactUs->service_id) {
                $serviceModel = \App\Models\Service::find($contactUs->service_id);
                $service = $serviceModel ? $serviceModel->name : 'غير محدد';
            }

            // Get product name if product_id exists
            $product = null;
            if ($contactUs->product_id) {
                $productModel = \App\Models\Product::find($contactUs->product_id);
                $product = $productModel ? $productModel->name : null;
            }

            // Build HTML email content
            $htmlContent = $this->buildEmailTemplate($name, $phone, $email, $company, $service, $product, $message);

            // Get notification emails (can be multiple)
            $notificationEmails = $this->getNotificationEmails($adminEmail);

            // Send email
            Mail::html($htmlContent, function ($mail) use ($notificationEmails, $name) {
                $mail->to($notificationEmails)
                    ->subject("📩 حجز جديد من موقع DHI - {$name}");
            });
        } catch (\Exception $e) {
            // Log error but don't throw - we don't want email failure to break contact form
            Log::error('Failed to send contact form notification email', [
                'error' => $e->getMessage(),
                'contact_id' => $contactUs->id ?? null
            ]);
        }
    }

    /**
     * Get list of emails to receive notifications
     * You can add multiple emails here
     */
    protected function getNotificationEmails($primaryEmail)
    {
        // الإيميلات اللي هتستقبل التنبيهات
        $emails = [$primaryEmail]; // الإيميل الأساسي من Settings

        // ⬇️ ضيف الإيميلات الإضافية هنا ⬇️
        $emails[] = 'ahmed.essam.be@gmail.com';
        // $emails[] = 'support@dhiegypt.com';
        // $emails[] = 'manager@dhiegypt.com';

        // إزالة الإيميلات المكررة والفارغة
        $emails = array_filter(array_unique($emails));

        return $emails;
    }
    /**
     * Build beautiful HTML email template
     */
    protected function buildEmailTemplate($name, $phone, $email, $company, $service, $product, $message)
    {
        // Build product field if exists
        $productField = '';
        if ($product) {
            $productField = "
            <div class='field'>
                <div class='label'>📦 المنتج:</div>
                <div class='value'>{$product}</div>
            </div>";
        }

        return "<!DOCTYPE html>
<html dir='rtl' lang='ar'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <style>
        body { 
            font-family: Arial, sans-serif; 
            direction: rtl; 
            margin: 0;
            padding: 0;
            background-color: #f4f4f4;
        }
        .container { 
            max-width: 600px; 
            margin: 20px auto; 
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .header { 
            background: #164852; 
            color: white; 
            padding: 30px 20px; 
            text-align: center; 
        }
        .header h2 {
            margin: 0;
            font-size: 24px;
        }
        .content { 
            background: #f5f5f5; 
            padding: 30px 20px; 
        }
        .field { 
            margin: 15px 0; 
            padding: 15px; 
            background: white; 
            border-right: 4px solid #7FC3C2; 
            border-radius: 4px;
        }
        .label { 
            font-weight: bold; 
            color: #164852; 
            font-size: 14px;
            margin-bottom: 8px;
        }
        .value { 
            color: #333; 
            font-size: 16px;
            line-height: 1.6;
        }
        .footer {
            background: #f9f9f9;
            padding: 20px;
            text-align: center;
            color: #666;
            font-size: 12px;
            border-top: 1px solid #e0e0e0;
        }
    </style>
</head>
<body>
    <div class='container'>
        <div class='header'>
            <h2>📩 رسالة جديدة من موقع DHI</h2>
        </div>
        <div class='content'>
            <div class='field'>
                <div class='label'>👤 الاسم:</div>
                <div class='value'>{$name}</div>
            </div>
            <div class='field'>
                <div class='label'>📧 البريد الإلكتروني:</div>
                <div class='value'>{$email}</div>
            </div>
            <div class='field'>
                <div class='label'>📱 رقم الهاتف:</div>
                <div class='value'>{$phone}</div>
            </div>
            <div class='field'>
                <div class='label'>🏢 الشركة:</div>
                <div class='value'>{$company}</div>
            </div>
            <div class='field'>
                <div class='label'>�️ الخدمة:</div>
                <div class='value'>{$service}</div>
            </div>
            {$productField}
            <div class='field'>
                <div class='label'>� الرسالة:</div>
                <div class='value'>" . nl2br(htmlspecialchars($message)) . "</div>
            </div>
            <div class='field'>
                <div class='label'>🕐 تاريخ الإرسال:</div>
                <div class='value'>" . date('Y-m-d H:i:s') . "</div>
            </div>
        </div>
        <div class='footer'>
            <p>هذه رسالة تلقائية من نظام إدارة المحتوى - DHI</p>
            <p>© " . date('Y') . " جميع الحقوق محفوظة</p>
        </div>
    </div>
</body>
</html>";
    }
}