<?php

namespace App\Services\Dashboard;

use App\Helper\Media;
use App\Services\JsonTranslationService;
use App\Models\Product;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

class ProductService
{
    /**
     * Clean Arabic text for slug usage
     * Keep Arabic characters but remove special characters and spaces
     */
    private function cleanArabicSlug($text)
    {
        // Replace spaces and special characters with dashes, but keep Arabic characters
        $slug = str_replace([' ', '/', '\\', '?', '#', '[', ']', '@', '!', '$', '&', "'", '(', ')', '*', '+', ',', ';', '=', '%'], '-', $text);

        // Remove multiple dashes
        $slug = preg_replace('/-+/', '-', $slug);

        // Remove leading/trailing dashes
        $slug = trim($slug, '-');

        return $slug;
    }

    public function store($request)
    {
        $data = $request->validated();

        DB::beginTransaction();
        try {
            // Handle file uploads
            if ($request->hasFile('image')) {
                $data['image'] = Media::uploadAndAttachImageStorage($request->file('image'), 'products');
            }
            if ($request->hasFile('icon')) {
                $data['icon'] = Media::uploadAndAttachImageStorage($request->file('icon'), 'products');
            }

            // Prepare main model data
            $mainData = $data;

            // Get translation fields
            $translationFields = JsonTranslationService::getTranslationFields('product');

            // Create model with JSON translations
            $model = JsonTranslationService::createWithTranslations(Product::class, $mainData, $request, $translationFields);

            DB::commit();
            return $model;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function update($request, $model)
    {
        $data = $request->validated();

        DB::beginTransaction();

        try {
            // Handle file uploads
            if ($request->hasFile('image')) {
                if ($model->image) {
                    Media::removeFile('products', $model->image);
                }
                $data['image'] = Media::uploadAndAttachImageStorage($request->file('image'), 'products');
            }

            if ($request->hasFile('icon')) {
                if ($model->icon) {
                    Media::removeFile('products', $model->icon);
                }
                $data['icon'] = Media::uploadAndAttachImageStorage($request->file('icon'), 'products');
            }

            // Prepare main model data
            $mainData = $data;

            // Get translation fields
            $translationFields = JsonTranslationService::getTranslationFields('product');

            // Update model with JSON translations
            $model = JsonTranslationService::updateWithTranslations($model, $mainData, $request, $translationFields);

            DB::commit();
            return $model;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function delete($selectedIds)
    {
        Log::info('Starting delete in ProductService', [
            'selectedIds' => $selectedIds
        ]);

        $products = Product::whereIn('id', $selectedIds)->get();

        Log::info('Found products to delete', [
            'count' => $products->count(),
            'products' => $products->pluck('id')->toArray()
        ]);

        DB::beginTransaction();
        try {
            foreach ($products as $product) {
                Log::info('Processing product', [
                    'product_id' => $product->id,
                    'has_image' => !empty($product->image),
                    'has_icon' => !empty($product->icon)
                ]);

                // Delete associated product images
                $this->deleteProductImages($product);

                // Delete associated image if it exists
                if ($product->image) {
                    try {
                        Media::removeFile('products', $product->image);
                        Log::info('Deleted product image', [
                            'product_id' => $product->id,
                            'image' => $product->image
                        ]);
                    } catch (\Exception $e) {
                        Log::warning('Failed to delete product image', [
                            'product_id' => $product->id,
                            'image' => $product->image,
                            'error' => $e->getMessage()
                        ]);
                    }
                }

                // Delete associated Icon if it exists
                if ($product->icon) {
                    try {
                        Media::removeFile('products', $product->icon);
                        Log::info('Deleted product icon', [
                            'product_id' => $product->id,
                            'icon' => $product->icon
                        ]);
                    } catch (\Exception $e) {
                        Log::warning('Failed to delete product icon', [
                            'product_id' => $product->id,
                            'icon' => $product->icon,
                            'error' => $e->getMessage()
                        ]);
                    }
                }

                // Delete the product model (this triggers the observer)
                try {
                    $product->delete();
                    Log::info('Deleted product', [
                        'product_id' => $product->id
                    ]);
                } catch (\Exception $e) {
                    Log::error('Failed to delete product', [
                        'product_id' => $product->id,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    throw $e;
                }
            }

            DB::commit();
            Log::info('Delete operation completed successfully');
            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Delete operation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    /**
     * Delete all images associated with a product
     */
    private function deleteProductImages(Product $product): void
    {
        try {
            $productImages = $product->productImages;

            foreach ($productImages as $image) {
                // Get the raw filename from the image
                $filename = $image->getImageFilenameAttribute();

                if ($filename) {
                    Media::removeFile('products', $filename);
                    Log::info('Deleted product image file', [
                        'product_id' => $product->id,
                        'image_id' => $image->id,
                        'filename' => $filename
                    ]);
                }

                $image->delete();
            }

            Log::info('Deleted all product images', [
                'product_id' => $product->id,
                'count' => $productImages->count()
            ]);
        } catch (\Exception $e) {
            Log::warning('Failed to delete some product images', [
                'product_id' => $product->id,
                'error' => $e->getMessage()
            ]);
        }
    }
}
