<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use App\Services\ImageResizeService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class ManifestController extends Controller
{
    private ImageResizeService $imageResizeService;

    public function __construct(ImageResizeService $imageResizeService)
    {
        $this->imageResizeService = $imageResizeService;
    }
    /**
     * Generate dynamic manifest.json
     */
    public function manifest(): JsonResponse
    {
        Log::info('Manifest controller called');

        // Get current locale
        $currentLocale = app()->getLocale();

        // Get site settings from database (general settings)
        $settings = Setting::where('lang', 'all')->pluck('value', 'key')->toArray();

        // Get PWA settings for current language
        $pwaSettings = Setting::whereIn('key', ['site_short_name', 'site_description', 'theme_color', 'background_color', 'site_logo'])
            ->where('lang', $currentLocale)
            ->pluck('value', 'key')
            ->toArray();

        // Get site name and description from current locale settings
        $siteName = $pwaSettings['site_name'] ?? config("configrations.site_name") ?? ($currentLocale === 'ar' ? 'موقعي' : 'My Website');
        $siteDescription = $pwaSettings['site_description'] ?? config("configrations.site_description") ?? 'Professional web development, SEO, and digital marketing services in Egypt';
        $siteShortName = $pwaSettings['site_short_name'] ?? config("configrations.site_name") ?? 'Tulip';

        // Get theme colors
        $themeColor = $pwaSettings['theme_color'] ?? $settings['theme_color'] ?? '#007bff';
        $backgroundColor = $pwaSettings['background_color'] ?? $settings['background_color'] ?? '#ffffff';

        // Get dynamic logo from database
        $logoUrl = $this->getDynamicLogo();

        // Get current locale
        $currentLocale = app()->getLocale();
        $isRTL = $currentLocale === 'ar';

        // Generate shortcuts based on available pages/menus
        $shortcuts = $this->generateShortcuts($currentLocale);

        // Generate icons array
        $icons = $this->generateIcons($logoUrl);

        // Get current base URL dynamically
        $baseUrl = url('/');
        $currentPath = request()->getPathInfo();

        // Extract subdirectory automatically from the current URL
        $subdirectory = '';
        $parsedUrl = parse_url($baseUrl);
        $path = $parsedUrl['path'] ?? '';

        // Remove trailing slash and get the subdirectory
        $path = rtrim($path, '/');
        if ($path && $path !== '/') {
            $subdirectory = $path;
        }

        $manifest = [
            'name' => $siteName,
            'short_name' => $siteShortName,
            'description' => $siteDescription,
            'start_url' => $subdirectory . '/' . $currentLocale,
            'display' => 'standalone',
            'background_color' => $backgroundColor,
            'theme_color' => $themeColor,
            'orientation' => 'portrait-primary',
            'icons' => $icons,
            'categories' => [
                'business',
                'productivity',
                'utilities'
            ],
            'lang' => $currentLocale,
            'dir' => $isRTL ? 'rtl' : 'ltr',
            'scope' => $subdirectory . '/',
            'prefer_related_applications' => false,
            'related_applications' => [],
            'screenshots' => [],
            'shortcuts' => $shortcuts
        ];

        return response()->json($manifest)
            ->header('Content-Type', 'application/manifest+json')
            ->header('Cache-Control', 'no-cache, no-store, must-revalidate') // Force refresh
            ->header('Pragma', 'no-cache')
            ->header('Expires', '0');
    }

    /**
     * Generate shortcuts based on available menus/pages
     */
    private function generateShortcuts(string $locale): array
    {
        $shortcuts = [];

        // Get current base URL dynamically
        $baseUrl = url('/');
        $parsedUrl = parse_url($baseUrl);
        $path = $parsedUrl['path'] ?? '';

        // Remove trailing slash and get the subdirectory
        $path = rtrim($path, '/');
        $subdirectory = ($path && $path !== '/') ? $path : '';

        // Get 96x96 icon for shortcuts
        $shortcutIcon = $this->getShortcutIcon();

        // Home shortcut
        $shortcuts[] = [
            'name' => $locale === 'ar' ? 'الصفحة الرئيسية' : 'Home',
            'short_name' => $locale === 'ar' ? 'الرئيسية' : 'Home',
            'description' => $locale === 'ar' ? 'الصفحة الرئيسية للموقع' : 'Website Homepage',
            'url' => $subdirectory . '/' . $locale,
            'icons' => [$shortcutIcon]
        ];

        // Services shortcut
        $shortcuts[] = [
            'name' => $locale === 'ar' ? 'الخدمات' : 'Services',
            'short_name' => $locale === 'ar' ? 'الخدمات' : 'Services',
            'description' => $locale === 'ar' ? 'خدماتنا المتاحة' : 'Our Services',
            'url' => $subdirectory . '/' . $locale . '/services',
            'icons' => [$shortcutIcon]
        ];

        // Products shortcut
        $shortcuts[] = [
            'name' => $locale === 'ar' ? 'المنتجات' : 'Products',
            'short_name' => $locale === 'ar' ? 'المنتجات' : 'Products',
            'description' => $locale === 'ar' ? 'منتجاتنا المتاحة' : 'Our Products',
            'url' => $subdirectory . '/' . $locale . '/products',
            'icons' => [$shortcutIcon]
        ];

        // Contact shortcut
        $shortcuts[] = [
            'name' => $locale === 'ar' ? 'اتصل بنا' : 'Contact Us',
            'short_name' => $locale === 'ar' ? 'اتصل بنا' : 'Contact',
            'description' => $locale === 'ar' ? 'تواصل معنا' : 'Get in Touch',
            'url' => $subdirectory . '/' . $locale . '/contact',
            'icons' => [$shortcutIcon]
        ];

        return $shortcuts;
    }

    /**
     * Get shortcut icon (96x96)
     */
    private function getShortcutIcon(): array
    {
        // Use the dynamic logo for shortcuts too
        $logoUrl = $this->getDynamicLogo();
        $logoExtension = pathinfo($logoUrl, PATHINFO_EXTENSION);
        $logoMimeType = $this->getMimeTypeFromExtension($logoExtension);

        return [
            'src' => $logoUrl,
            'sizes' => '96x96',
            'type' => $logoMimeType
        ];
    }
    /**
     * Get dynamic logo from database settings using Path helper
     */
    private function getDynamicLogo(): string
    {
        // Use the same logic as the website's Path::AppLogo helper
        return \App\Helper\Path::AppLogo('site_logo');
    }
    /**
     * Generate icons array based on available logo
     */
    private function generateIcons(string $logoUrl): array
    {
        $icons = [];

        // Add favicon as fallback
        $icons[] = [
            'src' => asset('favicon.ico'),
            'sizes' => '16x16 32x32 48x48',
            'type' => 'image/x-icon'
        ];

        // Try to generate square icons first
        $squareIcons = $this->generateSquareIconsUrls($logoUrl);

        if (!empty($squareIcons)) {
            // Use generated square icons
            foreach ($squareIcons as $size => $iconUrl) {
                $icons[] = [
                    'src' => $iconUrl,
                    'sizes' => "{$size}x{$size}",
                    'type' => 'image/png',
                    'purpose' => 'any'
                ];

                $icons[] = [
                    'src' => $iconUrl,
                    'sizes' => "{$size}x{$size}",
                    'type' => 'image/png',
                    'purpose' => 'maskable'
                ];
            }
        } else {
            // Fallback: Add specific sizes that Safari iOS supports
            $logoExtension = pathinfo($logoUrl, PATHINFO_EXTENSION);
            $logoMimeType = $this->getMimeTypeFromExtension($logoExtension);

            // Safari iOS requires specific icon sizes
            $safariSizes = [
                '120x120',  // iPhone app icon
                '152x152',  // iPad app icon  
                '167x167',  // iPad Pro app icon
                '180x180',  // iPhone 6 Plus app icon
                '192x192',  // Standard PWA icon
                '512x512'   // High-res PWA icon
            ];

            foreach ($safariSizes as $size) {
                $icons[] = [
                    'src' => $logoUrl,
                    'sizes' => $size,
                    'type' => $logoMimeType,
                    'purpose' => 'any'
                ];
            }

            // Add maskable version for Android
            $icons[] = [
                'src' => $logoUrl,
                'sizes' => '192x192',
                'type' => $logoMimeType,
                'purpose' => 'maskable'
            ];
        }
        return $icons;
    }

    /**
     * Generate square icons using ImageResizeService
     */
    private function generateSquareIconsUrls(string $logoUrl): array
    {
        try {
            // Check if logo URL is valid
            if (empty($logoUrl) || !filter_var($logoUrl, FILTER_VALIDATE_URL)) {
                Log::warning('Invalid logo URL for icon generation: ' . $logoUrl);
                return [];
            }

            $icons = $this->imageResizeService->generateSquareIcons($logoUrl, [152, 192, 512]);
            if (!empty($icons)) {
                return $icons;
            }
        } catch (\Exception $e) {
            // Log error but continue
            Log::warning('PWA Icon generation failed: ' . $e->getMessage());
        }

        return [];
    }
    /**
     * Get MIME type from file extension
     */
    private function getMimeTypeFromExtension(string $extension): string
    {
        $mimeTypes = [
            'svg' => 'image/svg+xml',
            'png' => 'image/png',
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'gif' => 'image/gif',
            'webp' => 'image/webp',
            'ico' => 'image/x-icon'
        ];

        return $mimeTypes[strtolower($extension)] ?? 'image/png';
    }
}
