<?php

namespace App\Models;

use App\Helpers\Helper;
use App\Models\ProductAttribute;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Maatwebsite\Excel\Concerns\ToArray;
use Spatie\Sitemap\Contracts\Sitemapable;
use Spatie\Sitemap\Tags\Url;

class Product extends Model implements Sitemapable
{
    use HasFactory;
    protected $table = 'products';
    private $lang;

    protected $fillable = [
        'name_ar',
        'name_en',
        'category_id',
        'brand_id',
        'code',
        'viewing_count',
        'order',
        'price',
        'quantity',
        'text_ar',
        'text_en',
        'short_text_ar',
        'short_text_en',
        'main_image',
        'second_image',
        'icon',
        'status',
        'link_ar',
        'link_en',
        'meta_title_ar',
        'meta_title_en',
        'meta_description_ar',
        'meta_description_en',
        'index',
        'home',
        'featured',
    ];

    public function __construct(array $attributes = [])
    {
        parent::__construct($attributes);

        // Set the current locale dynamically
        $this->lang = Helper::getLang();
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($product) {
            if (empty($product->code)) {
                $product->code = self::generateProductCode();
            }
        });
    }

    public function delete()
    {
        $errors = [];
        if ($this->stocks()->exists()) {
            foreach ($this->stocks as  $stock) {
                if ($stock->orders()->exists()) {
                    $errors[] = __('home.cannot delete an product that has related orders');
                }
            }
        }

        if (count($errors)) {
            return $errors;
        }
        return parent::delete();
    }

    public static function generateProductCode()
    {
        $latestProduct = self::orderBy('id', 'desc')->first();
        $latestId = $latestProduct ? $latestProduct->id : 0;
        return config('site_perfix') . str_pad($latestId + 1, 4, '0', STR_PAD_LEFT);
    }
    public function getNameAttribute()
    {
        return $this->{'name_' . $this->lang};
    }

    public function getTextAttribute()
    {
        return $this->{'text_' . $this->lang};
    }

    public function getLinkAttribute()
    {
        return $this->{'link_' . $this->lang};
    }

    public function getShortTextAttribute()
    {
        return $this->{'short_text_' . $this->lang};
    }

    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    public function getBrandNameAttribute()
    {
        return $this->brand?->name;
    }

    public function getBrandTextAttribute()
    {
        return $this->brand?->text;
    }

    public function getBrandLinkAttribute()
    {
        return $this->brand?->link;
    }

    public function getCategoryNameAttribute()
    {
        return $this->category?->name;
    }

    public function getCategoryTextAttribute()
    {
        return $this->category?->text;
    }

    public function getCategoryLinkAttribute()
    {
        return $this->category?->link;
    }


    public function getActiveAttribute()
    {
        $this->status ? __('home.yes') : __('home.no');
    }

    public function scopeActive($query)
    {
        $query->whereStatus(1);
    }
    public function scopeHome($query)
    {
        $query->whereHome(1);
    }

    public function scopeUnactive($query)
    {
        $query->whereStatus(0);
    }
    public function scopeFeatured($query)
    {
        $query->whereFeatured(1);
    }


    public function stocks()
    {
        return $this->hasMany(ProductStock::class);
    }


    public function getDiscountIdAttribute()
    {
        return $this->discount?->id;
    }

    public function getDiscountTypeAttribute()
    {
        return $this->discount?->type;
    }

    public function getDiscountValueAttribute()
    {
        return $this->discount?->discount;
    }


    public function getAttributeValuesAttribute()
    {
        $product_Attributes_id = $this->attributes()->pluck('id')->ToArray();
        return ProductAttributeValue::whereIn('product_attribute_id', $product_Attributes_id)->get();
    }


    public function reviews($query)
    {
        return $this->hasMany(ProductReview::class);
    }

    public function getReviewsCountAttribute()
    {
        return $this->reviews->count();
    }



    //// new ///

    public function attributes()
    {
        return $this->hasMany(ProductAttribute::class);
    }


    public function variants()
    {
        return $this->hasMany(ProductVariant::class);
    }

    public function category()
    {
        return $this->belongsTo(Category::class);
    }


    // Check if product has variants
    public function getHasVariantsAttribute()
    {
        return $this->variants->isNotEmpty();
    }



    public function getRouteKeyName(): string
    {
        if (!request()->is('*admin*')) {
            return 'link_' . $this->lang;
        }
        return 'id';
    }

    public function getStockAttribute()
    {
        if ($this->has_variants) {
            return $this->variants->sum('stock');
        }
        return $this->attributes['quantity'];
    }


    public function getMainImagePathAttribute()
    {

        return $this->main_image ? asset('storage/products/' . $this->main_image) : Helper::noImage();
    }

    public function getSecondImagePathAttribute()
    {

        return $this->second_image ? asset('storage/products/' . $this->second_image) : $this->main_image_path;
    }

    public function images()
    {
        return $this->morphMany(Image::class, 'imageable');;
    }

    // Get available colors with stock
    public function getAvailableColorsAttribute()
    {
        return $this->variants->groupBy('color')->map(function ($variants) {
            return [
                'variant_id' => $variants->first()->id,
                'color' => $variants->first()->color,
                'image' => $variants->first()->first_image,
                'stock' => $variants->sum('stock'),
                'additional_price' => $variants->min('additional_price'),
            ];
        });
    }

    public function getDefaultVariantAttribute()
    {
        return $this->variants()
            ->with(['images', 'values.value.attribute'])
            ->first();
    }
    public function discounts()
    {
        return $this->hasMany(ProductDiscount::class);
    }



    /**
     * Get the original price for a product or specific variant
     */
    public function getOriginalPrice($variantId = null)
    {
        $basePrice = $this->price;

        if ($variantId) {
            $variant = $this->variants()->find($variantId);
            if ($variant && $variant->additional_price) {
                $basePrice += $variant->additional_price;
            }
        }

           return (int) round($basePrice);
    }

    /**
     * Get the effective price after applying discounts

     */
    public function getEffectivePrice($variantId = null)
    {
        $originalPrice = $this->getOriginalPrice($variantId);
        $discount = $this->getApplicableDiscount($variantId);

        if (!$discount) {
            return $originalPrice;
        }

        $discountedPrice = $this->calculateDiscountedPrice($originalPrice, $discount);

        return (int) round($discountedPrice);

    }


    /**
     * Calculate discounted price based on discount type

     */
    private function calculateDiscountedPrice($originalPrice, $discount)
    {
        switch ($discount->type) {
            case 'percentage':
                return $originalPrice - ($originalPrice * $discount->value / 100);
            case 'fixed':
            case 'amount':
                return $originalPrice - $discount->value;
            default:
                return $originalPrice;
        }
    }

    /**
     * Get the applicable discount for the product or variant

     */
    public function getApplicableDiscount($variantId = null)
    {
        $now = now();

        $baseQuery = $this->discounts()
            ->where('is_active', true)
            ->where(function ($q) use ($now) {
                $q->whereNull('starts_at')->orWhere('starts_at', '<=', $now);
            })
            ->where(function ($q) use ($now) {
                $q->whereNull('expires_at')->orWhere('expires_at', '>', $now);
            });

        // Check for variant-specific discounts first (higher priority)
        if ($variantId) {
            $variantDiscount = (clone $baseQuery)
                ->where('apply_to', 'variant')
                ->where('variant_id', $variantId)
                ->orderBy('value', 'desc')
                ->first();

            if ($variantDiscount) {
                return $variantDiscount;
            }
        }

        // Check for product-wide discounts
        $productDiscount = (clone $baseQuery)
            ->where('apply_to', 'product')
            ->orderBy('value', 'desc')
            ->first();

        return $productDiscount;
    }
    /**
     * Check if the product/variant is currently on sale
     */
    public function isOnSale($variantId = null)
    {
        return $this->getEffectivePrice($variantId) < $this->getOriginalPrice($variantId);
    }

    public function getDiscountAmount($variantId = null)
    {
        $originalPrice = $this->getOriginalPrice($variantId);
        $effectivePrice = $this->getEffectivePrice($variantId);
        return $originalPrice - $effectivePrice;
    }

     public function getDiscountPercentage($variantId = null)
    {
        $originalPrice = $this->getOriginalPrice($variantId);
        $discountAmount = $this->getDiscountAmount($variantId);

        if ($originalPrice > 0 && $discountAmount > 0) {
            return round(($discountAmount / $originalPrice) * 100, 2);
        }

        return 0;
    }


    public function getAvailableStock($variantId = null)
    {

        if ($variantId) {
            $variant = $this->variants()->find($variantId);
            return $variant ? $variant->stock : 0;
        }

        return $this->has_variants ? $this->variants()->sum('stock') :$this->quantity;
    }

    public function isInStock($variantId = null)
    {

        return $this->getAvailableStock($variantId) > 0;
    }

    public function updateStock($quantity, $variantId = null)
    {
        if ($variantId) {
            $variant = $this->variants()->find($variantId);
            if ($variant) {
                $variant->stock -= $quantity;
                $variant->save();
            }
        } else {
            $this->quantity -= $quantity;
            $this->save();
        }
    }

    public function toSitemapTag(): Url | string | array
    {
          // Return with fine-grained control:
          return Url::create(route('product.details', $this->link))
          ->setLastModificationDate(Carbon::create($this->updated_at))
          ->setChangeFrequency(Url::CHANGE_FREQUENCY_YEARLY)
          ->setPriority(0.5);
    }

}
