<?php

namespace App\Http\Controllers\website;

use App\Http\Controllers\Controller;
use App\Models\Attribute;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Traits\SeoTrait;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    use SeoTrait;
    public function getProducts()
    {
        $data['products'] = Product::with(['variants.values.value.attribute', 'variants.images'])
            ->active()
            ->paginate(12);
        // SEO
        $seo = $this->productsPageSeo();
        $data['schema'] = $seo['schema'] ?? null;
        $data['metatags'] = $seo['metatags'] ?? null;

        return view('website.products', $data);
    }
    public function getProductDetails(Product $product)
    {
        $product->update(['viewing_count' => $product->viewing_count + 1]);



        $product->load(['variants.values.value.attribute', 'attributes.attribute']);

        $data['relatedProducts']  = Product::with(['variants.values.value.attribute', 'variants.images'])
            ->where('category_id', $product->category_id)
            ->whereNot('id', $product->id)
            ->inRandomOrder()
            ->active()
            ->limit(6)
            ->get();

        $data['product'] = $product;

        // SEO
        list($data['schema'], $data['metatags']) =  $this->productPageSeo($product);

        return view('website.product-details', $data);
    }



    public function getVariantData(Request $request)
    {
        $productId = $request->input('product_id');
        $selectedValues = $request->input('variant');

        $product = Product::with(['variants.values.value.attribute', 'attributes.attribute'])
            ->findOrFail($productId);

        // Get the variant that matches all selected values
        $variant = ProductVariant::with(['product', 'values.value.attribute', 'images'])
            ->where('product_id', $productId)
            ->where(function ($query) use ($selectedValues) {
                foreach ($selectedValues as $attributeId => $valueId) {
                    $query->whereHas('values', function ($q) use ($attributeId, $valueId) {
                        $q->where('value_id', $valueId)
                            ->whereHas('value', function ($v) use ($attributeId) {
                                $v->where('attribute_id', $attributeId);
                            });
                    });
                }
            })
            ->first();

        if (!$variant) {
            return response()->json([
                'error' => 'Variant not found'
            ], 404);
        }

        return response()->json([
            'variant' => [
                'id' => $variant->id,
                'currency' => config('app.currency'),
                'price' => $product->getOriginalPrice($variant->id),
                'is_on_sale' => $product->isOnSale($variant->id),
                'sale_price' => $product->getEffectivePrice($variant->id),
                'discount_amount' => $product->getDiscountAmount($variant->id),
                'in_stock' => $product->isInStock($variant->id),
                'images' => $variant->images->map(function ($image) {
                    return [
                        'src' => $image->imagePath('products/variants', $image->image),
                        'alt' => $image->alt_text ?? 'Variant image'
                    ];
                })->toArray(),
                'attributes' => $variant->values->mapWithKeys(function ($value) {
                    return [$value->value->attribute->name => $value->value->value];
                })->toArray()
            ]
        ]);
    }
}
