<?php

namespace App\Http\Controllers\website;

use App\Http\Controllers\Controller;
use App\Http\Requests\Website\PlaceOrderRequest;
use App\Mail\OrderSummaryMail;
use App\Models\ProductStock;
use App\Services\PaymobPaymentService;
use App\Traits\Carts\CartTrait;
use App\Traits\Carts\GuestCartTrait;
use App\Traits\Checkout\CouponTrait;
use App\Traits\PlaceOrder\GetOrCreateAddressTrait;
use App\Traits\PlaceOrder\HandleGuestUserTrait;
use App\Traits\PlaceOrder\OrderProcessTrait;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class PlaceOrderController extends Controller
{
    use CartTrait, GuestCartTrait, HandleGuestUserTrait, GetOrCreateAddressTrait, OrderProcessTrait, CouponTrait;

    protected $paymobPaymentService;

    // public function __construct(PaymobPaymentService $paymobPaymentService)
    // {
    //     $this->paymobPaymentService = $paymobPaymentService;
    // }

    public function __invoke(PlaceOrderRequest $request)
    {
dd($request->all());
        $data = $request->validated();
dd($data);
        DB::beginTransaction();

        try {
            // Step 1: Get the cart
            $cart = auth()->check() ? $this->getCart() : $this->getGuestCart();


            // Step 2: Get or create user ID
            $userId = auth()->check() ? auth()->id() : $this->execute($data)->id;


            // Step 3: Get or create address
            $address = $this->executeAddress($userId, $data);

            // Step 4: Create the order
            $order = $this->createOrder($userId, $address->id, $data,$cart , $address);


            // Step 5: Process order items
            $this->processOrderItems($cart, $order);


            // Step 6: Create order status
            $this->createOrderStatus($order);

            // Step 7: Apply coupon
            $this->couponUsed($cart, $userId, $order);


            // Step 8: Prepare items for Paymob
            $orderItems = $order->products->map(function ($orderItem) {
                $productStock = ProductStock::find($orderItem['product_stock_id']);
                $unitPriceCents = (int) round($orderItem['price_after_discount'] * 100);

                return [
                    'name' => $productStock->product->name,
                    'amount' => $unitPriceCents, // Store UNIT price here
                    'quantity' => $orderItem['quantity'],
                    'description' => $productStock->product->short_text_en ?? $productStock->product->name,
                ];
            })->toArray();

            // Calculate total correctly
            $calculatedTotal = collect($orderItems)->sum(function($item) {
                return $item['amount'] * $item['quantity']; // Multiply unit price × quantity
            });

            $amountCents = (int) round($order->total_price * 100);

            $shippingFee = $order->shipping_fees ?? 0.00;

            if ($shippingFee > 0) {
                $shippingItem = [
                    'name' => 'Shipping Fee',
                    'amount' => (int) round($shippingFee * 100),
                    'quantity' => 1,
                    'description' => 'Delivery service charge'
                ];

                $orderItems[] = $shippingItem;
                $calculatedTotal += $shippingItem['amount']; // Add to total
            }

            $billingData = [
                'apartment' => 'NA',
                'email' => auth()->user()->email ?? 'email not found!',
                'floor' => 'NA',
                'first_name' => auth()->user()->f_name,
                'street' => $order->address->address ?? 'NA',
                'building' => 'NA',
                'phone_number' => $order->address->phone ?? Auth::user()->phone,
                'postal_code' => 'NA',
                'city' => $order->address->Governorate->name ?? 'NA',
                'country' => $order->address->country->name ?? 'NA',
                'last_name' => Auth::user()->l_name ?? 'NA',
                'state' => 'NA'
            ];

            // Step 9: Commit transaction **before** payment processing
            DB::commit();


            // Step 10: Process payment if payment_id is 2 (Paymob)

            if ($data['payment_id'] == 2) {
                try {
                    $response = $this->paymobPaymentService->intention($amountCents, $orderItems, $billingData, $order->id);

                    if ($response instanceof \Illuminate\Http\JsonResponse && $response->status() == 500) {
                        return redirect()->back()->withErrors(['error' => 'Payment failed. Try again.']);
                    }

                    DB::commit(); // ✅ Commit بعد التأكد من نجاح الدفع

                    $cart->delete();
                } catch (\Exception $paymentException) {

                    Log::error('Payment failed: ' . $paymentException->getMessage());
                    DB::rollBack();
                    return redirect()->back()->withErrors(['error' => 'Payment failed. Try again.']);
                }
            }


            // Step 11: Delete the cart
            $cart->delete();


            // Send order summary email to the user
            if( $data['email']){
                Mail::to($order->user->email)->send(new OrderSummaryMail($order));
            }

            // You can also send the email to the admin or any other recipient if needed
            Mail::to("info@abouhamdan.com")->send(new OrderSummaryMail($order));


            return view('website.order-thank',compact('order'));
        } catch (\Exception $e) {

            // Step 12: Rollback the transaction on failure
            DB::rollBack();
            Log::error('Order placement failed: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return redirect()->back()->withErrors(['error' => 'Order placement failed. Please try again later.']);
        }
    }
}
