<?php

namespace App\Http\Controllers\Admin;


use App\Http\Controllers\Controller;
use App\Http\Requests\AttributeRequest;
use App\Models\Attribute;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;


class AttributeController extends Controller
{

    public function __construct()
    {
        $this->middleware(['permission:attributes']);
    }


    public function index()
    {

        $attributes = Attribute::with('values')->get();
        return view('admin.attributes.index', compact('attributes'));
    }


    public function create()
    {
        return view('admin.attributes.create');
    }


    public function store(AttributeRequest $request)
    {
        try {
            DB::beginTransaction();

            $data = $request->validated();

            Attribute::create($data);

            DB::commit();
            return redirect()->route('attributes.index')->with('success', trans('home.your_item_added_successfully'));
        } catch (Exception $ex) {
            DB::rollBack();
            Log::error('Transaction failed', ['exception' => $ex]);
            return redirect()->back()->withErrors($ex->getMessage());
        }
    }



    public function edit(Attribute $attribute)
    {
        $attribute->load(['values' => function ($q) {
            $q->orderBy('id', 'desc');
        }]);
        return view('admin.attributes.edit', compact('attribute'));
    }


    public function update(AttributeRequest $request, Attribute $attribute)
    {

        $data = $request->validated();

        try {

            DB::beginTransaction();

            $data['status'] = $data['status'] ?? 0;

            $attribute->update($data);

            DB::commit();

            return redirect()->back()->with('success', trans('home.your_item_updated_successfully'), compact('attribute'));
        } catch (\Exception $ex) {
            DB::rollBack();
            Log::error('Transaction failed', ['exception' => $ex]);
            return redirect()->back()->withErrors($ex->getMessage());
        }
    }

    public function destroy(Request $request)
    {
       
        DB::beginTransaction();
        try {
            $selectedIds = $request->input('id');

            $request->validate([
                'id' => ['array', 'min:1'],
                'id.*' => ['exists:attributes,id']
            ]);

            Attribute::whereIn('id', $selectedIds)->delete();

            DB::commit();

            return redirect()->back()->with(['success' => __('home.your_items_deleted_successfully')]);
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with(['error' => __('home.failed_to_deleted_item')]);
        }
    }
}
