<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\View;
use App\View\Composers\SeoComposer;
use App\View\Composers\MenuComposer;
use App\View\Composers\FooterMenuComposer;
use App\Models\Phone;
use App\Models\Dashboard\Menu;
use App\Models\SiteAddress;
use App\Models\Section;
use App\Models\Service;
use App\Models\Project;
use App\Models\Product;
use App\Models\Blog;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;
use Illuminate\Support\Facades\Route;

class ViewComposerServiceProvider extends ServiceProvider
{
    public function boot(): void
    {
        // SEO Composer for main layout pages and website components
        View::composer(['layouts.app', 'Website.*', 'components.website.*'], SeoComposer::class);

        // Menu Composer for navigation components
        View::composer(['components.website.partials.mainmenu', 'components.website.partials.mobilemenu'], MenuComposer::class);

        // Mobile Menu Composer for contact info
        View::composer('components.website.partials.mobilemenu', function ($view) {
            $phones = Phone::active()->orderBy('order', 'asc')->orderBy('updated_at', 'desc')->get();
            $site_addresses = SiteAddress::active()->orderBy('order', 'asc')->orderBy('updated_at', 'desc')->get();

            // Get settings for social media links
            $socialMediaLinks = [
                'whatsapp' => config('settings.site_whatsapp') ? ' https://wa.me/' . ltrim(config('settings.site_whatsapp'), '+') : '#',
                'facebook' => config('settings.site_facebook') ?? '#',
                'twitter' => config('settings.site_twitter') ?? '#',
                'instagram' => config('settings.site_instagram') ?? '#',
                'youtube' => config('settings.site_youtube') ?? '#',
                'linkedin' => config('settings.site_linkedin') ?? '#',
                'tiktok' => config('settings.site_tiktok') ?? '#',
                'snapchat' => config('settings.site_snapchat') ?? '#',
                'pinterest' => config('settings.site_pinterest') ?? '#',
                'telegram' => config('settings.site_telegram') ?? '#',
            ];

            $view->with([
                'phones' => $phones,
                'site_addresses' => $site_addresses,
                'socialMediaLinks' => $socialMediaLinks
            ]);
        });

        // Footer Menu Composer for footer navigation (parent menus only)
        View::composer(['components.website.partials._footer'], FooterMenuComposer::class);

        View::composer('components.website.partials._header', function ($view) {
            // Force fresh data from database (no cache)
            // Get the first phone by order (lowest order number = highest priority)
            $phone = Phone::active()->orderBy('order', 'asc')->orderBy('updated_at', 'desc')->first();
            $phones = Phone::active()->orderBy('order', 'asc')->orderBy('updated_at', 'desc')->get();
            $site_addresses = SiteAddress::active()->orderBy('order', 'asc')->orderBy('updated_at', 'desc')->get();

            $socialMediaLinks = [
                'whatsapp' => config('settings.site_whatsapp') ? ' https://wa.me/' . ltrim(config('settings.site_whatsapp'), '+') : '#',
                'facebook' => config('settings.site_facebook') ?? '#',
                'twitter' => config('settings.site_twitter') ?? '#',
                'instagram' => config('settings.site_instagram') ?? '#',
                'youtube' => config('settings.site_youtube') ?? '#',
                'linkedin' => config('settings.site_linkedin') ?? '#',
                'tiktok' => config('settings.site_tiktok') ?? '#',
                'snapchat' => config('settings.site_snapchat') ?? '#',
                'pinterest' => config('settings.site_pinterest') ?? '#',
                'telegram' => config('settings.site_telegram') ?? '#',
            ];

            // Language switcher logic
            $currentLocale = app()->getLocale();
            $supportedLocales = LaravelLocalization::getSupportedLocales();

            // Prepare languages array with URLs
            $languages = collect($supportedLocales)->map(function ($locale, $code) {
                return (object)[
                    'code' => $code,
                    'name' => $locale['native'],
                    'url' => \App\Helpers\LocalizationHelper::getCurrentPageLocalizedUrl($code),
                ];
            })->values();

            $altLangLink = null;
            $route = Route::current();
            $routeParameters = $route ? $route->parameters() : [];
            $targetLang = $currentLocale === 'ar' ? 'en' : 'ar';
            $currentPath = $route ? $route->uri() : '';

            if ($route) {
                $slug = null;
                $model = null;
                $targetPath = null;

                if (str_contains($currentPath, 'services')) {
                    $slug = $routeParameters['slug'] ?? null;
                    $model = Service::class;
                    $targetPath = 'services';
                } elseif (str_contains($currentPath, 'projects')) {
                    $item = $routeParameters['project'] ?? null;
                    if ($item instanceof Project) {
                        $slug = $item->getTranslation('slug', $currentLocale);
                    } else {
                        $slug = $item;
                    }
                    $model = Project::class;
                    $targetPath = 'projects';
                } elseif (str_contains($currentPath, 'products')) {
                    $item = $routeParameters['product'] ?? null;
                    if ($item instanceof Product) {
                        $slug = $item->getTranslation('slug', $currentLocale);
                    } else {
                        $slug = $item;
                    }
                    $model = Product::class;
                    $targetPath = 'products';
                } elseif (str_contains($currentPath, 'blogs')) {
                    $slug = $routeParameters['slug'] ?? null;
                    $model = Blog::class;
                    $targetPath = 'blogs';
                }

                if ($slug && $model && $targetPath) {
                    // Decode URL-encoded Arabic characters before database search
                    $decodedSlug = urldecode($slug);
                    $item = $model::whereRaw("JSON_EXTRACT(slug, '$.{$currentLocale}') = ?", [$decodedSlug])->first();

                    // If not found, try to find by checking all languages
                    if (!$item) {
                        $items = $model::all();
                        foreach ($items as $i) {
                            foreach (['en', 'ar', 'fr'] as $lang) {
                                $itemSlug = $i->getTranslation('slug', $lang);
                                if ($itemSlug === $decodedSlug) {
                                    $item = $i;
                                    break 2;
                                }
                            }
                        }
                    }

                    if ($item && method_exists($item, 'getLocalizedUrl')) {
                        $altLangLink = $item->getLocalizedUrl($targetLang);
                    }
                }
            }

            if (!$altLangLink) {
                $altLangLink = \App\Helpers\LocalizationHelper::getCurrentPageLocalizedUrl($targetLang);
            }

            $view->with('headerPhone', $phone);
            $view->with('phones', $phones);
            $view->with('site_addresses', $site_addresses);
            $view->with('socialMediaLinks', $socialMediaLinks);
            $view->with('altLangLink', $altLangLink);
            $view->with('languages', $languages);
        });

        //Footer composer Socal Media Links & Phones & Site Addresses
        View::composer('components.website.partials._footer', function ($view) {
            $socialMediaLinks = [
                'whatsapp' => config('settings.site_whatsapp') ? 'https://wa.me/' . ltrim(config('settings.site_whatsapp'), '+') : '#',
                'facebook' => config('settings.site_facebook') ?? '#',
                'twitter' => config('settings.site_twitter') ?? '#',
                'instagram' => config('settings.site_instagram') ?? '#',
                'youtube' => config('settings.site_youtube') ?? '#',
                'linkedin' => config('settings.site_linkedin') ?? '#',
                'tiktok' => config('settings.site_tiktok') ?? '#',
                'snapchat' => config('settings.site_snapchat') ?? '#',
                'pinterest' => config('settings.site_pinterest') ?? '#',
                'telegram' => config('settings.site_telegram') ?? '#',
            ];

            // Force fresh data from database (no cache)
            // Get all active phones ordered by priority for footer display
            $phones = Phone::active()->orderBy('order', 'asc')->orderBy('updated_at', 'desc')->get();
            $site_addresses = SiteAddress::active()->orderBy('order', 'asc')->orderBy('updated_at', 'desc')->get();
            $view->with('socialMediaLinks', $socialMediaLinks);
            $view->with('phones', $phones);
            $view->with('site_addresses', $site_addresses);
        });

        //Banner composer for Social Media Links
        View::composer('Website.partials._banner', function ($view) {
            $socialMediaLinks = [
                'whatsapp' => config('settings.site_whatsapp') ? 'https://wa.me/' . ltrim(config('settings.site_whatsapp'), '+') : '#',
                'facebook' => config('settings.site_facebook') ?? '#',
                'twitter' => config('settings.site_twitter') ?? '#',
                'instagram' => config('settings.site_instagram') ?? '#',
                'youtube' => config('settings.site_youtube') ?? '#',
                'linkedin' => config('settings.site_linkedin') ?? '#',
                'tiktok' => config('settings.site_tiktok') ?? '#',
                'snapchat' => config('settings.site_snapchat') ?? '#',
                'pinterest' => config('settings.site_pinterest') ?? '#',
                'telegram' => config('settings.site_telegram') ?? '#',
            ];

            $view->with('socialMediaLinks', $socialMediaLinks);
        });
    }


    public function register(): void
    {
        //
    }
}
