<?php

namespace App\Services;

use App\Models\Order;
use App\Models\OrderShipment;
use App\Models\OrderShipmentStatus;
use App\Models\OrderStatus;
use App\Models\Payment;
use App\Models\SimpleRequestOrder;
use App\Models\User;
use App\Traits\Carts\CartTrait;
use App\Traits\Carts\GuestCartTrait;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class OrderService
{
    use CartTrait, GuestCartTrait;

    protected $data;
    protected $cart;
    public function __construct(array $data)
    {
        $this->data = $data;
        $this->cart = Auth::check() ? $this->getCart() : $this->getGuestCart();
    }
    public function placeOrder()
    {

        DB::beginTransaction();

        try {


            // Step 1: Get or create user
            $this->getOrCreateUser();

            // Step 2: Get or create address
            $address =  (new UserAddressService())->createAddress($this->data);

            // Step 3: Create the order
            $order = $this->createOrder($address);

            // Step 4: Process order items
            $this->processOrderItems($order);

            // Step 5: Process payment
            $this->processPayment($order, null, $this->data['payment_method_id']);
            // Step 6: 
            $this->orderShipment($order);
            // Step 6: Delete the cart

            $this->cart->delete(); // Clear the cart after order is placed

            DB::commit();

            return $order;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function getOrCreateUser()
    {
        if (Auth::check()) {

            return auth()->id();
        } else {

            $user =  User::firstOrCreate(
                [
                    'email' => $data['email'] ?? 'guest_' . uniqid() . '@example.com',
                ],
                [
                    'f_name' => $this->data['f_name'],
                    'l_name' => $this->data['l_name'],
                    'phone' =>  $this->data['phone'],
                    'password' => Hash::make('password'),
                    'is_guest' => 1,
                ]
            );
            Auth::login($user);

            return $user;
        }
    }

    private function createOrder($address)
    {

        $shipping_fees =  (new ShippingService)->fees(
            $this->cart,
            $address->governorate_id ?? null
        );

        $order =  Order::create([
            'user_id' => auth()->id(),
            'address_id' => $address->id,
            'status_id' => OrderStatus::first()->id,
            'total_amount' => $this->cart->total_price + $shipping_fees,
            'shipping_fees' => $shipping_fees,
        ]);



        return $order;
    }

    private function processOrderItems($order)
    {

        foreach ($this->cart->items as $item) {

            $order->items()->create([
                'order_id' => $order->id,
                'product_id' => $item->product_id,
                'product_variant_id' => $item->product_variant_id,
                'price' => $item->product->getEffectivePrice($item->product_variant_id),
                'quantity' => $item->quantity,
            ]);

            // Update product stock
            $item->product->updateStock($item->quantity, $item->product_variant_id);
        }
    }
    public function processPayment(Order $order = null, SimpleRequestOrder $simple_request_order = null, $paymentMethodId)
    {

        // Handle Cash on Delivery
        if ($paymentMethodId == 1) {
            $paymentData = [
                'order_id' => $order?->id,
                'simple_request_order_id' => $simple_request_order?->id,
                'user_id' => $order?->user_id ?? $simple_request_order?->user_id,
                'payment_method_id' => $paymentMethodId,
                'amount' => $order?->total_amount ?? $simple_request_order?->total_amount,
                'status' => 'unpaid',
                'processed_at' => now(),
            ];

            Payment::create($paymentData);
            return true;
        }

        // Handle Paymob
        if ($paymentMethodId == 2) {
            dd('Paymob payment processing not implemented yet.');
            // (new PaymobPaymentService())->intention($order);
        }

        return false;
    }


    private function orderShipment($order)
    {
        OrderShipment::create([
            'order_id' => $order->id,
            'order_shipment_status_id' => OrderShipmentStatus::first()->id,
        ]);
    }
}
