<?php

namespace App\Http\Controllers\website;

use App\Http\Controllers\Controller;
use App\Http\Requests\Website\SavecommercialRequest;
use App\Http\Requests\Website\SaveContactUsRequest;
use App\Http\Requests\Website\SaveInspectionRequest;
use App\Models\About;
use App\Models\AboutStruc;
use App\Models\AttributeValue;
use App\Models\Blog;
use App\Models\Branch;
use App\Models\Category;
use App\Models\CommercialRequest;
use App\Models\ContactUs;
use App\Models\Country;
use App\Models\Governorate;
use App\Models\InspectionRequest;
use App\Models\Page;
use App\Models\Product;
use App\Models\SiteAddress;
use App\Models\Subscribe;
use App\Services\FilterProductsService;
use App\Traits\SeoTrait;
use App\Traits\Website\Products\ProductFilterTrait;
use Illuminate\Http\Request;

use Illuminate\Support\Facades\Mail;


class WebsiteController extends Controller
{
    use SeoTrait, ProductFilterTrait;


    public function getCategories(Category $category = null)
    {

        // Show top-level categories if no specific category requested
        if (!$category) {

            $data['categories'] = Category::with('subcategories')
                ->whereNull('parent_id')
                ->active()
                ->get();

            // Add SEO data

            list($data['schema'], $data['metatags']) = $this->categoriesPageSeo();

            return view('website.categories', $data);
        }

        // Load the requested category with all nested subcategories
        $category->load('subcategories');

        // If category has subcategories, show them
        if ($category->subcategories->isNotEmpty()) {
            list($data['schema'], $data['metatags']) = $this->categoriesPageSeo();
            return view('website.subCategories', $data);
        } else {

            return redirect()->route('website.category.products', $category);
        }
    }

    public function getCategoryProducts(Request $request, Category $category, FilterProductsService $filterProductsService)
    {

        $data['category'] = $category;


        $data['products']  = $category->products()
            ->active()
            ->with(['variants.values.value.attribute', 'images', 'variants.images'])
            ->orderBy('created_at', 'desc')
            ->paginate(12);


        $data['other_categories'] = Category::where('id', '!=', $category->id)->get();

        // Get all products for price calculation (not paginated)
        $allProducts = $category->products()->active()->with('variants')->get();
        $prices = $allProducts->pluck('price')->toArray();
        $data['minPrice'] = min($prices);
        $data['maxPrice'] =  max($prices);

        // Get all unique color codes from the products in the category
        $allColorCodes = $category->products()
            ->active()
            ->with(['variants.values.value.attribute', 'images', 'variants.images'])
            ->get()
            ->flatMap(function ($product) {
                return $product->getAvailableColorsAttribute()->keys();
            })
            ->unique()
            ->values();

        $data['categoryColors'] = AttributeValue::whereIn('code', $allColorCodes)->get();


        // if ($request->has('filter_availability')) {
        //     $query = $filterProductsService->filter($request, $query);
        // }

        // $data['products'] = $query->paginate(12);
        // Add SEO data

        list($data['schema'], $data['metatags']) = $this->categoryPageSeo($category);
        return view('website.category-products', $data);
    }

    // public function categ
    public function getAboutUs()
    {
        $data['about'] = About::first();
        $data['about_structs'] = AboutStruc::active()->get();
        list($data['schema'], $data['metatags']) = $this->aboutUsPageSeo();
        return view('website.about-us', $data);
    }

    public function contactUs()
    {
        $data['site_addresses'] = SiteAddress::active()->get();
        // Add SEO data
        $seo = $this->contactUsPageSeo();
        $data['schema'] = $seo['schema'] ?? null;
        $data['metatags'] = $seo['metatags'] ?? null;

        return view('website.contact-us', $data);
    }

    public function saveContactUs(SaveContactUsRequest $request)
    {

        $validatedData = $request->validated();
        $contact =  ContactUs::create($validatedData);
        $data = array('contact' => $contact);
        // Mail::send('emails/contact_email', $data, function ($msg) {
        //     $msg->to(config('site_contact_email'), 'Abouhamdan')->subject('Contact Us');
        //     // $msg->cc(['ahmedabobasha10@gmail.com']);
        //     $msg->from(config('mail.from.address'), config('mail.from.name'));
        // });

        return redirect()->back()->with(['success' => __('Thank you for contacting us. A customer service officer will contact you soon')]);
    }

    public function commercialRequest()
    {

        $data['countries'] = Country::where('status', 1)->get();

        $data['governorates'] = Governorate::where('status', true)->get();

        // Add SEO data
        list($data['schema'], $data['metatags']) = $this->commercialRequestPageSeo();


        return view('website.commercial-request', $data);
    }

    public function saveCommercialRequest(SavecommercialRequest $request)
    {
        // Validate and save the commercial request
        $validatedData = $request->validated();

        $commercialRequest = CommercialRequest::create($validatedData);


        return redirect()->back()->with(['success' => __('home.your_item_added_successfully')]);
    }


    public function inspectionRequest()
    {
        $data['countries'] = Country::where('status', 1)->get();
        $data['governorates'] = Governorate::where('status', true)->get();

        // Add SEO data
        list($data['schema'], $data['metatags']) =  $this->inspectionRequestPageSeo();

        return view('website.inspection-request', $data);
    }

    public function saveInspectionRequest(SaveInspectionRequest $request)
    {
        // Validate and save the inspection request
        $validatedData = $request->validated();
        $inspectionRequest = InspectionRequest::create($validatedData);

        return redirect()->back()->with(['success' => __('home.your_item_added_successfully')]);
    }

    public function getQuickViewModal(Request $request)
    {

        // Retrieve product ID from the request
        $productId = $request->input('productId');

        // Fetch the product by ID
        $product = Product::findOrFail($productId);  // Using findOrFail to handle if product is not found

        // Render the modal content as a view and return it as a response
        $html = view('website.modals.quick_view_modal', compact('product'))->render();

        return response()->json([
            'success' => true,
            'html' => $html,  // Return the modal HTML
        ]);
    }


    public function getPage(Request $request, $link)
    {
        $page  = Page::where('link_en', $link)->orWhere('link_ar', $link)->first();

        // seo data
        $seo = $this->getPageSeo($page);
        $data['schema'] = $seo['schema'] ?? null;
        $data['metatags'] = $seo['metatags'] ?? null;



        return view('website.page', compact('page'));
    }

    public function getBranches()
    {
        $data['branches'] = Branch::with(['governorate'])
            ->where('status', true)
            ->orderBy('created_at', 'desc')
            ->get();

        $data['governorates'] = Governorate::whereHas('branches', function ($query) {
            $query->where('status', true);
        })->get();

        // Add SEO data
        // $seo = $this->branchesPageSeo();
        $data['schema'] = $seo['schema'] ?? null;
        $data['metatags'] = $seo['metatags'] ?? null;

        return view('website.branches', $data);
    }

    public function getBranchesFilter(Request $request)
    {
        $query = Branch::with(['governorate']);

        if ($request->has('governorate_id') && $request->governorate_id) {
            $query->where('governorate_id', $request->governorate_id);
        }

        $branches = $query->get();

        $html = view('website._branches_append', compact('branches'))->render();

        return response()->json(['html' => $html]);
    }

    public function getMoodboards()
    {
        // Add SEO data
        // $seo = $this->moodboardsPageSeo();
        $data['schema'] = $seo['schema'] ?? null;
        $data['metatags'] = $seo['metatags'] ?? null;
        return view('website.moodboards');
    }

    public function subscribe(Request $request)
    {
        $request->validate([
            'email' => 'required|email|unique:subscribes,email',
        ], [
            'email.required' => __('home.email_required'),
            'email.email' => __('home.email_invalid'),
            'email.unique' => __('home.email_already_subscribed'),
        ]);

        Subscribe::create(['email' => $request->email]);

        return redirect()->back()->with(['success' => __('home.subscription_successful')]);
    }

    public function getBlogs()
    {

        $data['blogs'] = Blog::active()
            ->orderBy('created_at', 'desc')
            ->get();

        // seo data
        $seo = $this->blogsPageSeo();
        $data['schema'] = $seo['schema'] ?? null;
        $data['metatags'] = $seo['metatags'] ?? null;

        return view('website.blogs', $data);
    }

    public function getBlogDetails(Blog $blog)
    {
        $data['blog'] = $blog;
        $data['other_blogs'] = Blog::active()->where('id', '!=', $blog->id)->orderBy('created_at', 'desc')->take(5)->get();
        list($data['schema'], $data['metatags']) =  $this->blogPageSeo($blog);

        return view('website.blog-details', $data);
    }
}
