<?php

namespace App\Http\Controllers\website;

use App\Http\Controllers\Controller;
use App\Http\Requests\Website\PlaceSampleRequest;
use App\Models\Country;
use App\Models\Governorate;
use App\Models\OrderShipment;
use App\Models\OrderShipmentStatus;
use App\Models\OrderStatus;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\SampleRequest;
use App\Models\SimpleRequestOrder;
use App\Models\SimpleRequestOrderOrderItems;
use App\Services\OrderService;
use App\Services\UserAddressService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SampleRequestController extends Controller
{

    public function addTOSampleRequest(Request $request)
    {

        // Validate the request data
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'selectedVariantValues' => 'nullable'
        ]);

        try {

            $productId = $request->input('product_id');
            $variant_values = $request->input('selectedVariantValues');

            $product = Product::findOrFail($productId);

            if ($product->has_variants) {

                // 1. Find matching variant
                $matchingVariant = ProductVariant::where('product_id', $productId)
                    ->where(function ($query) use ($variant_values) {
                        foreach ($variant_values as $attributeId => $valueId) {
                            $query->whereHas('values', function ($q) use ($attributeId, $valueId) {
                                $q->where('value_id', $valueId)
                                    ->whereHas('value.attribute', function ($subQ) use ($attributeId) {
                                        $subQ->where('attribute_id', $attributeId);
                                    });
                            });
                        }
                    })
                    ->first();

                if (!$matchingVariant) {
                    return response()->json([
                        'status' => false,
                        'message' => 'No matching variant found for the selected options'
                    ], 404);
                }
            }



            if (Auth::check()) {

                $userId = auth()->user()->id;

                SampleRequest::updateOrCreate(
                    [
                        'user_id' => $userId,
                        'product_id' => $productId,
                        'product_variant_id' => $matchingVariant->id ?? null
                    ],
                    [
                        'user_id' => $userId,
                        'product_id' => $productId,
                        'product_variant_id' => $matchingVariant->id ?? null
                    ]
                );

                $SampleRequestCount = SampleRequest::where('user_id', $userId)->count();
            } else {


                $session_id = session()->getId();

                SampleRequest::updateOrCreate(
                    [
                        'session_id' => $session_id,
                        'product_id' => $productId,
                        'product_variant_id' => $matchingVariant->id ?? null
                    ],
                    [
                        'session_id' => $session_id,
                        'product_id' => $productId,
                        'product_variant_id' => $matchingVariant->id ?? null
                    ]
                );

                $SampleRequestCount = SampleRequest::where('session_id', $session_id)->count();
            }

            return response()->json([
                'status' => true,
                'SampleRequestCount' => $SampleRequestCount,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }

        return response()->json(['message' => 'Sample request added successfully!'], 201);
    }

    public function checkSampleRequest()
    {
        if (auth()->check()) {
            $data['sampleRequests'] = SampleRequest::with([
                'product',
                'variant.images',
                'variant.values',
                'variant.values.value',
                'variant.values.value.attribute',
                'variant.images'
            ])
                ->where('user_id', auth()->user()->id)
                ->get();
        } else {
            $session_id = session()->getId();

            $data['sampleRequests'] = SampleRequest::with([
                'product',
                'variant.images',
                'variant.values',
                'variant.values.value',
                'variant.values.value.attribute',
                'variant.images'
            ])
                ->where('session_id', $session_id)
                ->get();
        }

        if ($data['sampleRequests']->isEmpty()) {
            return redirect()->back()->with('error', __('home.no_sample_requests_found'));
        }

        $data['countries'] = Country::where('status', 1)->get();

        $data['governorates'] = Governorate::where('status', true)->get();



        return view('website.sample-request-checkout', $data);
    }

    public function placeSampleOrder(PlaceSampleRequest $request, UserAddressService $userAddressService,)
    {
        DB::beginTransaction();

        try {

            $data = $request->validated();

            $originalSessionId = session()->getId();

            $userId  =  (new OrderService($data))->getOrCreateUser($data);

            $address = $userAddressService->createAddress($data);


            $shipping_fees = $address->governorate->simple_request_shipping_fees ?? 0.00;

            // Get sample requests based on auth or session
           
            if (auth()->check()) {
                // if user was previously guest, retrieve by the saved old session
                $sampleRequests = SampleRequest::where(function ($q) use ($userId, $originalSessionId) {
                    $q->where('user_id', $userId)
                        ->orWhere('session_id', $originalSessionId); // 💡 include old session ID
                })->get();
            } else {
                $sampleRequests = SampleRequest::where('session_id', $originalSessionId)->get();
            }

           
            if ($sampleRequests->isEmpty()) {
                return redirect()->back()->with('error', __('website.no_sample_requests_found'));
            }

            // Create the sample order
            $order = SimpleRequestOrder::create([
                'user_id' => $userId,
                'address_id' => $address->id,
                'shipping_fees' => $shipping_fees,
                'note' => $data['note'] ?? null,
                'status_id' => OrderStatus::first()->id,
            ]);


            // Create order items
            foreach ($sampleRequests as $item) {
                SimpleRequestOrderOrderItems::create([
                    'order_id' => $order->id,
                    'product_id' => $item->product_id,
                    'product_variant_id' => $item->product_variant_id, // corrected field name
                ]);
            }

            (new OrderService($data))->processPayment(null, $order, $data['payment_method_id']);

            OrderShipment::create([
                'simple_request_order_id' => $order->id,
                'order_shipment_status_id' => OrderShipmentStatus::first()->id,
            ]);

            // Delete old sample requests
            $sampleRequests->each->delete();

            DB::commit();

            return view('website.order-thank');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', __('website.error_placing_order') . ' ' . $e->getMessage());
        }
    }
}
