<?php

namespace App\Http\Controllers\website;

use App\Http\Controllers\Controller;
use App\Models\CartItems;
use App\Models\PaymentMethod;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Traits\Carts\CartTrait;
use App\Traits\Carts\GuestCartTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;


class CartController extends Controller
{
    use CartTrait, GuestCartTrait;
    public function addToCart(Request $request)
    {
        $request->validate([
            'productId' => 'required|exists:products,id',
            'quantity' => 'nullable|integer|min:1',
            'variants' => 'nullable'
        ]);

        try {
            $productId = $request->input('productId');
            $quantity = $request->input('quantity');
            $variant_values = $request->input('variants');

            $product = Product::findOrFail($productId);

            if ($product->has_variants) {

                // 1. Find matching variant
                $matchingVariant = ProductVariant::where('product_id', $productId)
                    ->where(function ($query) use ($variant_values) {
                        foreach ($variant_values as $attributeId => $valueId) {
                            $query->whereHas('values', function ($q) use ($attributeId, $valueId) {
                                $q->where('value_id', $valueId)
                                    ->whereHas('value.attribute', function ($subQ) use ($attributeId) {
                                        $subQ->where('attribute_id', $attributeId);
                                    });
                            });
                        }
                    })
                    ->first();

                if (!$matchingVariant) {
                    return response()->json([
                        'status' => false,
                        'message' => 'No matching variant found for the selected options'
                    ], 404);
                }
            }

            if ($product->getAvailableStock($matchingVariant->id ?? null) < $quantity) {

                return response()->json([
                    'status' => false,
                    'message' => "Only {$product->getAvailableStock($matchingVariant->id ?? null)} items available in stock."
                ], 400);
            }

            // 2. Get or create cart
            $cart = Auth::check() ? $this->getCart() : $this->getGuestCart();

            // 3. Add to cart - using proper increment method
            $cartItem = CartItems::firstOrNew([
                'cart_id' => $cart->id,
                'product_id' => $productId,
                'product_variant_id' => $matchingVariant->id ?? null
            ]);

            $cartItem->quantity = $cartItem->exists ? $cartItem->quantity + $quantity : $quantity;
            $cartItem->save();

            // 4. Return response
            $cart = $cart->load([
                'items',
                'items.product',
                'items.variant',
                'items.variant.values',
                'items.variant.values.value',
                'items.variant.values.value.attribute',
                'items.variant.images'
            ]);
            
            $cart->refresh(); 
            $cartItemsCount = $cart->items->count();

            $cart = $cart->load([
                'items',
                'items.product',
                'items.variant',
                'items.variant.values',
                'items.variant.values.value',
                'items.variant.values.value.attribute',
                'items.variant.images'
            ]);

            $html = view('website.layouts.modals.partials.shoppingCart-append', [
                'cart' => $cart,
            ])->render();

            return response()->json([
                'status' => true,
                'message' => 'Item added to cart successfully.',
                'html' => $html,
                'cartItemsCount' => $cartItemsCount,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }

    public function UpdateQuantity(Request $request)
    {
        $cart = Auth::check() ? $this->getCart() : $this->getGuestCart();

        $cartItemId = $request->input('cart_item_id');
        $newQuantity = $request->input('quantity');

        $cartItem = CartItems::where('id', $cartItemId)
            ->where('cart_id', $cart->id)
            ->firstOrFail();

        if ($cartItem->productStock->stock >= $newQuantity) {
            $cartItem->quantity = $newQuantity;
            $cartItem->save();

            // Recalculate cart totals
            $cart->load('items');

            $totalPrice = $cart->items->sum(function ($item) {
                return $item->price * $item->quantity;
            });

            $cart->update(['products_price' => $totalPrice, 'total_price' => $totalPrice]);
            $cart->refresh();
            $cartItems = CartItems::where('cart_id', $cart->id)->get();
            return response()->json([
                'status' => true,
                'cart_total' => $totalPrice,
                'currency' => config('app.currency'),
                // this for view-cartItems blade
                'cartItem' => $cartItem->total_item_price,
                'html' => view('website.modals.shoppingCart-append', compact('cart', 'cartItems'))->render(),

            ]);
        }
        return response()->json(['status' => false, 'cartItemQuantity' => $cartItem->quantity]);
    }
    public function shoppingCart()
    {


        if (Auth::check()) {

            $userId = auth()->user()->id;

            $cart = $this->getCart();
        } else {

            $cart = $this->getGuestCart();
        }

        if ($cart->items->count() == 0) {
            return redirect('/');
        }

        $payment_logos = PaymentMethod::select('logo','name_ar','name_en')
            ->where('status', 1)
            ->get();


        return view('website.shopping-cart', compact('cart', 'payment_logos'));
    }


    public function removeCartItem(Request $request)
    {
        $cart = Auth::check() ? $this->getCart() : $this->getGuestCart();

        $cartItemId = $request->input('cartItemId');

        $cartItem = CartItems::with(['product', 'variant.values.value.attribute', 'variant.images'])
            ->where('id', $cartItemId)
            ->where('cart_id', $cart->id)
            ->firstOrFail();



        $cartItem->delete();

        // Reload the latest cart items from the database
       $cart->load(['items.product', 'items.variant.values.value.attribute', 'items.variant.images']);



        $cartItemsCount = $cart->items->count();


        return response()->json([
            'status' => true,
            'totalPrice' => $cart->total_price,
            'cartItemsCount' => $cartItemsCount,
            'currency' => config('app.currency'),
            'html' => view('website.layouts.modals.partials.shoppingCart-append', compact('cart'))->render(),
        ]);
    }
}
