<?php

namespace App\Http\Controllers\Admin\Products;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\Products\DeleteProductRequest;
use App\Http\Requests\Admin\Products\StoreProductRequest;
use App\Http\Requests\Admin\Products\UpdateProductRequest;
use App\Models\Product;
use App\Models\Category;
use App\Models\Attribute;
use App\Models\Brand;
use App\Services\ProductServices\ProductService;

class ProductController extends Controller
{

    public function __construct()
    {

        $this->middleware(['permission:products']);
    }


    public function index()
    {
        $products = Product::with(['category', 'brand', 'variants','discounts'])
            ->orderBy('id', 'DESC')
            ->get();
        return view('admin.products.index', compact('products'));
    }

    public function create()
    {
        $data['categories'] = Category::where('status', 1)
            ->whereDoesntHave('subcategories')
            ->get();
        $data['brands'] = Brand::where('status', 1)->get();
        $data['attributes'] = Attribute::where('status', 1)
            ->whereHas('values')
            ->get();
        return view('admin.products.create', $data);
    }

    public function store(StoreProductRequest $request, ProductService $service)
    {
        try {

            $product =  $service->store($request);

            if ($product->attributes()->exists()) {

                return redirect()->route('products.variants.index', $product->id);
            } else {
                return redirect()->back()->with('success', __('home.your_items_added_successfully'));
            }
        } catch (\Exception $e) {

            return redirect()->back()->with('error', $e->getMessage());

        }
    }

    public function edit(Product $product)
    {
        $data['product'] = $product;
        $data['categories'] = Category::where('status', 1)->whereDoesntHave('subcategories')->get();
        $data['brands'] = Brand::where('status', 1)->get();
        $data['attributes'] = Attribute::where('status', 1)->whereHas('values')->get();

        return view('admin.products.edit', $data);
      
    }

    public function update(UpdateProductRequest $request, $product, ProductService $service)
    {
        try {
            $product = Product::findOrFail($product);
            $hasNewAttributes = $service->update($request, $product);
            
            return $hasNewAttributes
                ? redirect()->route('products.variants.index', $product->id)
                    ->with('success', __('home.your_item_updated_successfully'))
                : redirect()->route('products.index')
                    ->with('success', __('home.your_item_updated_successfully'));
                    
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
    

    public function destroy(DeleteProductRequest $request , ProductService $service)
    {
        $selectedIds = $request->input('id');

        $request->validated();

        $deleted = $service->delete($selectedIds);


        if (request()->ajax()) {
            if (!$deleted) {
                return response()->json(['message' => $deleted ?? __('home.an messages.error entering data')], 422);
            }
            return response()->json(['success' => true, 'message' => __('home.your_items_deleted_successfully')]);
        }
        if (!$deleted) {
            return redirect()->back()->withErrors($delete ?? __('home.an error has occurred. Please contact the developer to resolve the issue'));
        }
    }
}
