<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\Media;
use App\Http\Controllers\Controller;
use App\Http\Requests\StoreBlogRequest;
use App\Http\Requests\UpdateBlogRequest;
use App\Models\Blog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
class BlogController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $data['blogs'] = Blog::all();

        return view('admin.blogs.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.blogs.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreBlogRequest $request)
    {
        try {
            $data = $request->validated();
            $data['slug_en'] = Str::slug($data['slug_en']);
            $data['slug_ar'] = str_replace(' ', '-', trim($data['slug_ar']));

            if ($request->has('image')) {
                $data['image'] = Media::uploadAndAttachImage($request->file('image'), 'blogs');
            }
            Blog::create($data);

            return redirect()->route('blogs.index')->with('success', 'Blog created successfully.');
        } catch (\Exception $e) {

            return redirect()->back()->with('error', 'An error occurred while saving the blog.' . $e->getMessage());
        }
    }



    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Blog $blog)
    {
        $data['blogItem'] = $blog;
        return view('admin.blogs.edit', $data);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateBlogRequest $request, Blog $blog)
    {
        try{
            $data = $request->validated(); 

            $data['slug_en'] = Str::slug($data['slug_en']);
            $data['slug_ar'] = str_replace(' ', '-', trim($data['slug_ar']));
            
            if ($request->has('image')) {
                if ($blog->image) {
                    Media::removeFile('blogs', $blog->image);
                }
                $data['image'] = Media::uploadAndAttachImage($request->file('image'), 'blogs');
            }
            $blog->update($data);


            return redirect()->route('blogs.index')->with('success', 'Blog updated successfully.');

        }catch (\Exception $e) {
            return redirect()->back()->with('error', 'An error occurred while updating the blog.' . $e->getMessage());
        }
        
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request)
    {

        $selectedIds = $request->input('id');

        $blogs = Blog::whereIn('id', $selectedIds)->get();

        DB::beginTransaction();
        try {
            foreach ($blogs as $blog) {
                // Delete associated image if it exists
                if ($blog->image) {
                    Media::removeFile('blogs', $blog->image);
                }
               
            }
            Blog::whereIn('id', $selectedIds)->delete();

            DB::commit();

            return response()->json(['success' => true, 'message' => __('home.your_items_deleted_successfully')]);
        } catch (\Exception $e) {

            DB::rollBack();
            return response()->json(['error' => true, 'message' => __('home.an error has occurred. Please contact the developer to resolve the issue')]);
        }
    }
}
