<?php

namespace App\Services\Dashboard;

use App\Helper\Media;
use App\Models\Service as ModelsService;
use App\Models\ServiceImage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

class Service
{
    /**
     * Create a new class instance.
     */
   public function store($request , $data)
   {
    DB::beginTransaction();

        try {
            // Generate slugs
            $data['slug_ar'] = preg_replace('/\s+/u', '-', trim($data['name_ar']));
            $data['slug_ar'] = preg_replace('/[^\p{L}\p{N}_-]+/u', '', $data['slug_ar']);
            $data['slug_en'] = Str::slug($data['name_en']);

            if ($request->hasFile('image')) {
                $data['image'] = Media::uploadAndAttachImageStorage($request->file('image'), 'services');
            }

            if($request->hasFile('icon')) {
                $data['icon'] = Media::uploadAndAttachImageStorage($request->file('icon'), 'services');
            }

            // Create the Service
            $service = ModelsService::create($data);

            // Handle service images upload if any
            if ($request->hasFile('service_images')) {
                $this->handleServiceImagesUpload($request->file('service_images'), $service->id);
            }

            DB::commit();
            Log::info('Service created successfully', ['service_id' => $service->id]);

            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create service: ' . $e->getMessage());
            throw $e;
        }
   }

   public function update($request, $data, $service)
   {
       DB::beginTransaction();
       try{
           $data['status'] = $data['status'] ?? 0;
           $data['show_in_home'] = $data['show_in_home'] ?? 0;
           $data['show_in_header'] = $data['show_in_header'] ?? 0;
           $data['show_in_footer'] = $data['show_in_footer'] ?? 0;
           $data['index'] = $data['index'] ?? 0;
           $data['slug_ar'] = preg_replace('/\s+/u', '-', trim($data['slug_ar']));
           $data['slug_ar'] = preg_replace('/[^\p{L}\p{N}_-]+/u', '', $data['slug_ar']);
           $data['slug_en'] = Str::slug($data['slug_en']);
           
           // تحديث صورة الأيقونة فقط إذا تم رفع أيقونة جديدة
           if ($request->hasFile('icon')) {
               if ($service->icon) {
                   Media::removeFile('services', $service->icon);
               }
               $data['icon'] = Media::uploadAndAttachImageStorage($request->file('icon'), 'services');
           }
           
           // تحديث صورة الخدمة فقط إذا تم رفع صورة جديدة
           if ($request->hasFile('image')) {
               if ($service->image) {
                   Media::removeFile('services', $service->image);
               }
               $data['image'] = Media::uploadAndAttachImageStorage($request->file('image'), 'services');
           }
           
           $service->Update($data);
           
           // Handle additional service images upload if any
           if ($request->hasFile('service_images')) {
               $this->handleServiceImagesUpload($request->file('service_images'), $service->id);
           }
           
           DB::commit();
           Log::info('Service updated successfully', ['service_id' => $service->id]);
           return true;
       }catch (\Exception $e) {
           DB::rollBack();
           Log::error('Failed to update service: ' . $e->getMessage());
           throw $e;
       }
   }

   public function delete($selectedIds) 
   {
       Log::info('Starting delete in DashboardService', [
           'selectedIds' => $selectedIds
       ]);

       $services = ModelsService::whereIn('id', $selectedIds)->get();
       
       Log::info('Found services to delete', [
           'count' => $services->count(),
           'services' => $services->pluck('id')->toArray()
       ]);
   
       DB::beginTransaction();
       try {
           foreach ($services as $service) {
               Log::info('Processing service', [
                   'service_id' => $service->id,
                   'has_image' => !empty($service->image),
                   'has_icon' => !empty($service->icon)
               ]);

               // Delete associated image if it exists
               if ($service->image) {
                   try {
                       Media::removeFile('services', $service->image);
                       Log::info('Deleted service image', [
                           'service_id' => $service->id,
                           'image' => $service->image
                       ]);
                   } catch (\Exception $e) {
                       Log::warning('Failed to delete service image', [
                           'service_id' => $service->id,
                           'image' => $service->image,
                           'error' => $e->getMessage()
                       ]);
                   }
               }
   
               // Delete associated Icon if it exists
               if ($service->icon) {
                   try {
                       Media::removeFile('services', $service->icon);
                       Log::info('Deleted service icon', [
                           'service_id' => $service->id,
                           'icon' => $service->icon
                       ]);
                   } catch (\Exception $e) {
                       Log::warning('Failed to delete service icon', [
                           'service_id' => $service->id,
                           'icon' => $service->icon,
                           'error' => $e->getMessage()
                       ]);
                   }
               }

               // Delete associated service images
               $this->deleteServiceImages($service);
   
               // Delete the service model (this triggers the observer)
               try {
                   $service->delete();
                   Log::info('Deleted service', [
                       'service_id' => $service->id
                   ]);
               } catch (\Exception $e) {
                   Log::error('Failed to delete service', [
                       'service_id' => $service->id,
                       'error' => $e->getMessage(),
                       'trace' => $e->getTraceAsString()
                   ]);
                   throw $e;
               }
           }
   
           DB::commit();
           Log::info('Delete operation completed successfully');
           return true;
   
       } catch (\Exception $e) {
           DB::rollBack();
           Log::error('Delete operation failed', [
               'error' => $e->getMessage(),
               'trace' => $e->getTraceAsString()
           ]);
           throw $e;
       }
   }

   /**
    * Handle service images upload
    */
   private function handleServiceImagesUpload($images, int $serviceId): void
   {
       try {
           $maxOrder = ServiceImage::where('service_id', $serviceId)->max('order') ?? 0;
           foreach ($images as $image) {
               $filename = Media::uploadAndAttachImageStorage($image, 'services');
               ServiceImage::create([
                   'service_id' => $serviceId,
                   'image' => $filename,
                   'order' => ++$maxOrder,
               ]);
               Log::info('Service image uploaded successfully', [
                   'service_id' => $serviceId,
                   'filename' => $filename,
                   'order' => $maxOrder
               ]);
           }
       } catch (\Exception $e) {
           Log::error('Error uploading service images: ' . $e->getMessage());
           throw $e;
       }
   }

   /**
    * Delete all service images for a service
    */
   private function deleteServiceImages(ModelsService $service): void
   {
       try {
           $serviceImages = $service->serviceImages;
           foreach ($serviceImages as $image) {
               $filename = $image->getImageFilenameAttribute();
               if ($filename) {
                   Media::removeFile('services', $filename);
                   Log::info('Deleted service image file', [
                       'service_id' => $service->id,
                       'image_id' => $image->id,
                       'filename' => $filename
                   ]);
               }
               $image->delete();
           }
           Log::info('Deleted all service images', [
               'service_id' => $service->id,
               'count' => $serviceImages->count()
           ]);
       } catch (\Exception $e) {
           Log::warning('Failed to delete some service images', [
               'service_id' => $service->id,
               'error' => $e->getMessage()
           ]);
       }
   }
}
