<?php

namespace App\Services\Dashboard;

use App\Helper\Media;
use App\Models\Project as ModelsProject;
use App\Models\ProjectImage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

class ProjectService extends BaseDashboardService
{
    protected function getModel(): string
    {
        return ModelsProject::class;
    }

    protected function getImagePath(): string
    {
        return 'projects';
    }

    public function store($request)
    {
        DB::beginTransaction();
        try {
            $data = $request->validated();
            Log::info('Creating new project', ['data' => array_diff_key($data, ['image', 'icon', 'project_images'])]);

            // Generate slugs
            $data['slug_ar'] = preg_replace('/\s+/u', '-', trim($data['name_ar']));
            $data['slug_ar'] = preg_replace('/[^\p{L}\p{N}_-]+/u', '', $data['slug_ar']);
            $data['slug_en'] = Str::slug($data['name_en']);

            if ($request->hasFile('image')) {
                $data['image'] = Media::uploadAndAttachImageStorage($request->file('image'), 'projects');
            }

            if ($request->hasFile('icon')) {
                $data['icon'] = Media::uploadAndAttachImageStorage($request->file('icon'), 'projects');
            }

            $project = ModelsProject::create($data);

            // Handle project images upload if any
            if ($request->hasFile('project_images')) {
                $this->handleProjectImagesUpload($request->file('project_images'), $project->id);
            }

            DB::commit();
            Log::info('Project created successfully', ['project_id' => $project->id]);
            return $project;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create project: ' . $e->getMessage());
            throw $e;
        }
    }

    public function update($request, $data, $project)
    {
        DB::beginTransaction();
        try {
            Log::info('Starting project update', [
                'project_id' => $project->id,
                'has_image_file' => $request->hasFile('image'),
                'has_icon_file' => $request->hasFile('icon'),
                'has_project_images' => $request->hasFile('project_images'),
                'image_files_count' => $request->hasFile('project_images') ? count($request->file('project_images')) : 0
            ]);

            $data['status'] = $data['status'] ?? 0;
            $data['show_in_home'] = $data['show_in_home'] ?? 0;
            $data['show_in_header'] = $data['show_in_header'] ?? 0;
            $data['show_in_footer'] = $data['show_in_footer'] ?? 0;
            $data['index'] = $data['index'] ?? 0;
            $data['slug_ar'] = preg_replace('/\s+/u', '-', trim($data['slug_ar']));
            $data['slug_ar'] = preg_replace('/[^\p{L}\p{N}_-]+/u', '', $data['slug_ar']);
            $data['slug_en'] = Str::slug($data['slug_en']);

            if ($request->hasFile('icon')) {
                Log::info('Processing icon file upload', ['project_id' => $project->id]);
                if ($project->icon) {
                    Media::removeFile('projects', $project->icon);
                }
                $data['icon'] = Media::uploadAndAttachImageStorage($request->file('icon'), 'projects');
                Log::info('Icon file uploaded successfully', ['project_id' => $project->id, 'filename' => $data['icon']]);
            }

            if ($request->hasFile('image')) {
                Log::info('Processing image file upload', ['project_id' => $project->id]);
                if ($project->image) {
                    Media::removeFile('projects', $project->image);
                }
                $data['image'] = Media::uploadAndAttachImageStorage($request->file('image'), 'projects');
                Log::info('Image file uploaded successfully', ['project_id' => $project->id, 'filename' => $data['image']]);
            }

            Log::info('Updating project with data', ['project_id' => $project->id, 'data_keys' => array_keys($data)]);
            $project->update($data);

            // Handle additional project images upload if any
            if ($request->hasFile('project_images')) {
                Log::info('Processing project images upload', [
                    'project_id' => $project->id,
                    'images_count' => count($request->file('project_images'))
                ]);
                $this->handleProjectImagesUpload($request->file('project_images'), $project->id);
            }

            DB::commit();
            Log::info('Project updated successfully', ['project_id' => $project->id]);
            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to update project: ' . $e->getMessage(), [
                'project_id' => $project->id,
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    public function delete($selectedIds)
    {
        Log::info('Starting delete in ProjectService', [
            'selectedIds' => $selectedIds
        ]);

        $projects = ModelsProject::whereIn('id', $selectedIds)->get();

        Log::info('Found projects to delete', [
            'count' => $projects->count(),
            'projects' => $projects->pluck('id')->toArray()
        ]);

        DB::beginTransaction();
        try {
            foreach ($projects as $project) {
                Log::info('Processing project', [
                    'project_id' => $project->id,
                    'has_image' => !empty($project->image),
                    'has_icon' => !empty($project->icon)
                ]);

                // Delete associated image if it exists
                if ($project->image) {
                    try {
                        Media::removeFile('projects', $project->image);
                        Log::info('Deleted project image', [
                            'project_id' => $project->id,
                            'image' => $project->image
                        ]);
                    } catch (\Exception $e) {
                        Log::warning('Failed to delete project image', [
                            'project_id' => $project->id,
                            'image' => $project->image,
                            'error' => $e->getMessage()
                        ]);
                    }
                }

                // Delete associated Icon if it exists
                if ($project->icon) {
                    try {
                        Media::removeFile('projects', $project->icon);
                        Log::info('Deleted project icon', [
                            'project_id' => $project->id,
                            'icon' => $project->icon
                        ]);
                    } catch (\Exception $e) {
                        Log::warning('Failed to delete project icon', [
                            'project_id' => $project->id,
                            'icon' => $project->icon,
                            'error' => $e->getMessage()
                        ]);
                    }
                }

                // Delete associated project images
                $this->deleteProjectImages($project);

                // Delete the project model
                try {
                    $project->delete();
                    Log::info('Deleted project', [
                        'project_id' => $project->id
                    ]);
                } catch (\Exception $e) {
                    Log::error('Failed to delete project', [
                        'project_id' => $project->id,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    throw $e;
                }
            }

            DB::commit();
            Log::info('Delete operation completed successfully');
            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Delete operation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    /**
     * Handle project images upload
     */
    private function handleProjectImagesUpload($images, int $projectId): void
    {
        try {
            Log::info('Starting project images upload', [
                'project_id' => $projectId,
                'images_count' => is_array($images) ? count($images) : 1
            ]);

            $maxOrder = ProjectImage::where('project_id', $projectId)->max('order') ?? 0;

            foreach ($images as $index => $image) {
                Log::info('Processing image', [
                    'project_id' => $projectId,
                    'image_index' => $index,
                    'original_name' => $image->getClientOriginalName(),
                    'size' => $image->getSize(),
                    'mime_type' => $image->getMimeType()
                ]);

                $filename = Media::uploadAndAttachImageStorage($image, 'projects');

                $projectImage = ProjectImage::create([
                    'project_id' => $projectId,
                    'image' => $filename,
                    'order' => ++$maxOrder,
                ]);

                Log::info('Project image uploaded successfully', [
                    'project_id' => $projectId,
                    'image_id' => $projectImage->id,
                    'filename' => $filename,
                    'order' => $maxOrder
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Error uploading project images: ' . $e->getMessage(), [
                'project_id' => $projectId,
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    /**
     * Delete all project images for a project
     */
    private function deleteProjectImages(ModelsProject $project): void
    {
        try {
            $projectImages = $project->projectImages;
            foreach ($projectImages as $image) {
                $filename = $image->getImageFilenameAttribute();
                if ($filename) {
                    Media::removeFile('projects', $filename);
                    Log::info('Deleted project image file', [
                        'project_id' => $project->id,
                        'image_id' => $image->id,
                        'filename' => $filename
                    ]);
                }
                $image->delete();
            }
            Log::info('Deleted all project images', [
                'project_id' => $project->id,
                'count' => $projectImages->count()
            ]);
        } catch (\Exception $e) {
            Log::warning('Failed to delete some project images', [
                'project_id' => $project->id,
                'error' => $e->getMessage()
            ]);
        }
    }
}
