<?php

namespace App\Models;

use App\Traits\HasLanguage;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;

class Project extends Model
{
    use HasFactory, HasLanguage;
    protected $table = 'projects';
    protected $fillable = [
        'category_id',
        'parent_id',
        'name_en',
        'name_ar',
        'short_desc_en',
        'short_desc_ar',
        'long_desc_en',
        'long_desc_ar',
        'image',
        'alt_image',
        'icon',
        'alt_icon',
        'status',
        'show_in_home',
        'show_in_header',
        'show_in_footer',
        'meta_title_ar',
        'meta_title_en',
        'meta_desc_ar',
        'meta_desc_en',
        'index',
        'slug_en',
        'slug_ar',
        'order',
        'clients_en',
        'clients_ar',
        'location_en',
        'location_ar',
        'category_en',
        'category_ar',
        'service_en',
        'service_ar',
        'date',
    ];
    public function getRouteKeyName()
    {
        if (!request()->is('*dashboard*')) {
            return 'slug_' . app()->getLocale(); // for frontend
        }

        return 'id'; // for dashboard/admin
    }

    public function projectImages()
    {
        return $this->hasMany(ProjectImage::class, 'project_id')->orderBy('order');
    }

    public function images()
    {
        return $this->projectImages();
    }

    public function parent()
    {
        return $this->belongsTo(Project::class, 'parent_id');
    }
    public function children()
    {
        return $this->hasMany(Project::class, 'parent_id');
    }
    public function category()
    {
        return $this->belongsTo(Category::class);
    }
    public function getImagePathAttribute()
    {
        return $this->image ? asset('storage/projects/' . $this->image) : asset('assets/dashboard/images/noimage.png');
    }
    public function getIconPathAttribute()
    {
        return $this->icon ? asset('storage/projects/' . $this->icon) : asset('assets/dashboard/images/noimage.png');
    }
    public function getNameAttribute()
    {
        return $this->{'name_' . $this->lang};
    }
    public function getShortDescAttribute()
    {
        return $this->{'short_desc_' . $this->lang};
    }
    public function getLongDescAttribute()
    {
        return $this->{'long_desc_' . $this->lang};
    }
    public function scopeActive($query)
    {
        return $query->where('status', true);
    }
    public function scopeHome($query)
    {
        return $query->where('show_in_home', true);
    }

    public function scopeHeader($query)
    {
        return $query->where('show_in_header', true);
    }
    public function scopeFooter($query)
    {
        return $query->where('show_in_footer', true);
    }
    public function getMetaTitleAttribute()
    {
        return $this->{'meta_title_' . $this->lang};
    }
    public function getMetaDescAttribute()
    {
        return $this->{'meta_desc_' . $this->lang};
    }
    public function getLinkAttribute()
    {
        return  LaravelLocalization::localizeUrl('projects/' . $this->{'slug_' . $this->lang});
    }
    public function getClientsAttribute()
    {
        return $this->{'clients_' . $this->lang};
    }
    public function getlocationAttribute()
    {
        return $this->{'location_' . $this->lang};
    }
    public function getcategoryNameAttribute()
    {
        return $this->{'category_' . $this->lang};
    }
    public function getserviceAttribute()
    {
        return $this->{'service_' . $this->lang};
    }
}
