<?php

use Illuminate\Foundation\Application;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;
use Spatie\Permission\Middleware\PermissionMiddleware;
use Spatie\Permission\Middleware\RoleMiddleware;
use Spatie\Permission\Middleware\RoleOrPermissionMiddleware;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Validation\ValidationException;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;


return Application::configure(basePath: dirname(__DIR__))
    ->withRouting(function () {
        // Global routes (non-localized) - should come first for auth and redirects
        Route::middleware('web')
            ->group(base_path('routes/web.php'));

        // Website routes with localization
        Route::middleware(['web', 'localeSessionRedirect', 'localizationRedirect', 'localeViewPath', 'redirects', 'lang.redirect'])
            ->prefix(LaravelLocalization::setLocale())
            ->name('website.')
            ->group(base_path('routes/web/website/website.php'));

        Route::middleware(['web'])
            ->prefix(LaravelLocalization::setLocale().'/dashboard')
            ->name('dashboard.')
            ->group(base_path('routes/web/dashboard/auth.php'));
        Route::group(['prefix' => LaravelLocalization::setLocale(), 'middleware' => ['localeSessionRedirect', 'localizationRedirect', 'localeViewPath', 'redirects', 'lang.redirect', 'secure.headers']], function () {
            Route::middleware(['web', 'auth:admin'])
                ->prefix('dashboard')
                ->name('dashboard.')
                ->group(base_path('routes/web/dashboard/dashboard.php'));
        });

        Route::prefix('api')
            ->middleware([  'api', 'api.secret', 'api.lang'])
            ->group(base_path('routes/api.php'));
    })
    ->withMiddleware(function ($middleware) {

        // Redirect unauthenticated admin users to dashboard login
        $middleware->redirectGuestsTo(fn() => route('dashboard.login'));

        // Add no-cache middleware globally to prevent all caching issues
        $middleware->append(\App\Http\Middleware\NoCacheMiddleware::class);

        $middleware->alias([

            'localize'                => \Mcamara\LaravelLocalization\Middleware\LaravelLocalizationRoutes::class,
            'localizationRedirect'    => \Mcamara\LaravelLocalization\Middleware\LaravelLocalizationRedirectFilter::class,
            'localeSessionRedirect'   => \Mcamara\LaravelLocalization\Middleware\LocaleSessionRedirect::class,
            'localeCookieRedirect'    => \Mcamara\LaravelLocalization\Middleware\LocaleCookieRedirect::class,
            'localeViewPath'          => \Mcamara\LaravelLocalization\Middleware\LaravelLocalizationViewPath::class,
            'redirects'               => \App\Http\Middleware\RedirectsMiddleware::class,
            'lang.redirect'           => \App\Http\Middleware\LangRedirection::class,
            'secure.headers'          => \App\Http\Middleware\SecurityHeadersMiddleware::class,
            'no.cache'                => \App\Http\Middleware\NoCacheMiddleware::class,
            'role' => RoleMiddleware::class,
            'permission' => PermissionMiddleware::class,
            'role_or_permission' => RoleOrPermissionMiddleware::class,
            'api.secret' => \App\Http\Middleware\ApiSecretKey::class,
            'api.lang' => \App\Http\Middleware\ApiLang::class,

        ]);
    })
     ->withExceptions(function ($exceptions) {

        $exceptions->render(function (Throwable $e, Request $request) {
            Log::error($e->getMessage().' in '.$e->getFile().' at line '.$e->getLine());

            // Handle JSON / API responses
            if ($request->expectsJson() || $request->is('api/*')) {

                // Determine status code for this exception
                $status = match (true) {
                    $e instanceof ValidationException           => 422,
                    $e instanceof AuthenticationException       => 401,
                    $e instanceof AuthorizationException        => 403,
                    $e instanceof ModelNotFoundException,
                    $e instanceof NotFoundHttpException         => 404,
                    $e instanceof HttpException                 => $e->getStatusCode(),
                    default                                     => 500,
                };

                // Use real message for all — except 500 (depends on debug)
                $message = ($status === 500 && !config('app.debug'))
                    ? 'Something went wrong, please try again later.'
                    : $e->getMessage(). (!config('app.debug') ?'':'// at Page: '.$e->getFile().' line: '.$e->getLine());

                // Validation errors if exist
              $errors = null;
                if ($e instanceof ValidationException) {
                    $errors = $e->errors();
                }

                // Return unified JSON response
                return response()->json([
                    'success' => false,
                    'status'  => $status,
                    'message' => $message,
                    'errors'  => $errors,
                ], $status);
            }
        });
    })
    ->withCommands([
        \App\Console\Commands\SitemapGenerate::class,
        \App\Console\Commands\GenerateRobotsTxt::class,
    ])

    ->create();
