<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use App\Http\Requests\Dashboard\Features\DeleteFeatureRequest;
use App\Http\Requests\Dashboard\Features\StoreFeatureRequest;
use App\Http\Requests\Dashboard\Features\UpdateFeatureRequest;
use App\Models\Feature;
use App\Services\Dashboard\FeatureService;

class FeatureController extends Controller
{
    protected $service;
    public function __construct(FeatureService $service)
    {
        $this->service = $service;
    }
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $this->authorize('about_structs.view');
        $data['features'] = Feature::get();

        $data['section'] = 'features';
        return view('Dashboard.Features.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorize('about_structs.create');
        $data['section'] = 'features';
        return view('Dashboard.Features.create', $data);
    }

    /**
     * Display the specified resource.
     */
    public function show(Feature $feature)
    {
        $this->authorize('about_structs.view');
        $data['feature'] = $feature;
        $data['section'] = 'features';
        return view('Dashboard.Features.show', $data);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreFeatureRequest $request)
    {
        try {
            $this->authorize('about_structs.create');
            $this->service->store(request: $request);
            return redirect()->route('dashboard.features.index')->with('success', __('dashboard.your_items_added_successfully'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }


    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Feature  $feature)
    {
        $this->authorize('about_structs.edit');
        $data['feature'] = $feature;
        $data['section'] = 'features';
        return view('Dashboard.Features.edit', $data);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateFeatureRequest $request, Feature $feature)
    {
        try {
            $this->authorize('about_structs.edit');
            $this->service->update($request, $feature);
            return redirect()->route('dashboard.features.index')->with('success', __('dashboard.your_items_updated_successfully'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
    /**
     * Remove the specified resource from storage.
     */
    public function destroy(DeleteFeatureRequest $request, $feature)
    {
        $this->authorize('about_structs.delete');

        $selectedIds = $request->input('selectedIds', [$feature]);

        $deleted = $this->service->delete($selectedIds);


        if (request()->ajax()) {
            if (!$deleted) {
                return response()->json(['message' => __('dashboard.an_error_occurred')], 422);
            }
            return response()->json(['success' => true, 'message' => __('dashboard.your_items_deleted_successfully')]);
        }
        if (!$deleted) {
            return redirect()->back()->withErrors(__('dashboard.an_error_occurred'));
        }
    }

    /**
     * Bulk destroy features
     */
    public function bulkDestroy(DeleteFeatureRequest $request)
    {
        $this->authorize('about_structs.delete');

        $selectedIds = $request->input('selectedIds');

        $deleted = $this->service->delete($selectedIds);

        if (request()->ajax()) {
            if (!$deleted) {
                return response()->json(['message' => __('dashboard.an_error_occurred')], 422);
            }
            return response()->json(['success' => true, 'message' => __('dashboard.your_items_deleted_successfully')]);
        }
        if (!$deleted) {
            return redirect()->back()->withErrors(__('dashboard.an_error_occurred'));
        }
    }
}

